// $Id: CImageManager.cpp,v 1.12 2007/03/03 02:59:20 paul Exp $

/*
 * All contents of this source code are copyright 2005 Exp Digital Uk.
 * This source file is covered by the licence conditions of the Infinity API. You should have recieved a copy
 * with the source code. If you didnt, please refer to http://www.expdigital.co.uk
 * All content is the Intellectual property of Exp Digital Uk.
 * Certain sections of this code may come from other sources. They are credited where applicable.
 * If you have comments, suggestions or bug reports please visit http://support.expdigital.co.uk
 */

#include "CImageManager.hpp"
#include "CTGAImage.hpp"
#include "CJPEGImage.hpp"
#include "CPNGImage.hpp"
#include <IO/CTextStream.hpp>
#include <IO/CXMLDocument.hpp>
#include <Basics/CStringTokeniser.hpp>
#include <Testing/CTrace.hpp>
using Exponent::Testing::CTrace;
using Exponent::Basics::CStringTokeniser;
using Exponent::IO::CTextStream;
using Exponent::IO::CXMLDocument;
using Exponent::IO::CXMLNode;
using Exponent::IO::CXMLAttribute;
using Exponent::GUI::Graphics::CTGAImage;
using Exponent::GUI::Graphics::CImageManager;

//	===========================================================================
EXPONENT_CLASS_IMPLEMENTATION(CImageManager::CImageReference, CCountedObject);

//	===========================================================================
EXPONENT_CLASS_IMPLEMENTATION(CImageManager::CAnimationReference, CCountedObject);

//	===========================================================================
TCountedPointerArray<CImageManager::CImageReference> CImageManager::CIMAGE_MANAGER_IMAGES;

//	===========================================================================
TCountedPointerArray<CImageManager::CAnimationReference> CImageManager::CIMAGE_MANAGER_ANIMATIONS;

//	===========================================================================
CSystemString CImageManager::CIMAGE_MANAGER_DEFAULT_RESOURCE_PATH;

//	===========================================================================
const char CImageManager::CIMAGE_MANAGER_XML_TAG_ROOT[]							= "image_manager_root";
const char CImageManager::CIMAGE_MANAGER_XML_TAG_VERSION[]						= "image_manager_xml_version";
const char CImageManager::CIMAGE_MANAGER_XML_TAG_IMAGES[]						= "images";
const char CImageManager::CIMAGE_MANAGER_XML_TAG_ANIMATIONS[]					= "animations";
const char CImageManager::CIMAGE_MANAGER_XML_TAG_CHILD_IMAGE[]					= "image";
const char CImageManager::CIMAGE_MANAGER_XML_ATTRIBUTE_IMAGE_RELATIVE_PATH[]	= "relative_path";
const char CImageManager::CIMAGE_MANAGER_XML_ATTRIBUTE_IMAGE_FILENAME[]			= "filename";
const char CImageManager::CIMAGE_MANAGER_XML_ATTRIBUTE_IMAGE_NAME[]				= "reference_name";
const char CImageManager::CIMAGE_MANAGER_XML_ATTRIBUTE_ANIMATION_FRAME_HEIGHT[]	= "frame_height";
const char CImageManager::CIMAGE_MANAGER_XML_ATTRIBUTE_ANIMATION_NUM_FRAMES[]	= "number_of_frames";

//	===========================================================================
void CImageManager::setDefaultPathToImages(const CSystemString &defaultPath)
{
	CIMAGE_MANAGER_DEFAULT_RESOURCE_PATH = defaultPath;
}

//	===========================================================================
bool CImageManager::addImage(const CSystemString &path, const CString &filename, const CString &referenceName)
{
	try
	{
		// Create the full path
		CSystemString thePath = CIMAGE_MANAGER_DEFAULT_RESOURCE_PATH;

		// If the path is {} the user is saying that they want the default resource path
		if (CString("{}") != path)
		{
			thePath.appendPath(path);
		}

		// Replace the seperators
		thePath.replacePathSeperators();

		// Try to load the image
		IImage *theImage = CTGAImage::getNewInstance(thePath, filename);

		// Check its valid
		if (theImage == NULL)
		{
			return false;
		}

		// Stick it in the array
		CIMAGE_MANAGER_IMAGES.addElement(new CImageReference(theImage, referenceName));

		// Done :)
		return true;
	}
	catch(...)
	{
		return false;
	}
}

//	===========================================================================
bool CImageManager::addAnimation(const CSystemString &path, const CString &filename, const CString &referenceName, const long frameHeight, const long numberOfFrames)
{
	try
	{
		// Create the full path
		CSystemString thePath = CIMAGE_MANAGER_DEFAULT_RESOURCE_PATH;

		// If the path is {} the user is saying that they want the default resource path
		if (CString("{}") != path)
		{
			thePath.appendPath(path);
		}

		// Replace the seperators
		thePath.replacePathSeperators();

		// Try to load the image
		IImage *theImage = CTGAImage::getNewInstance(thePath, filename);

		// Check its valid
		if (theImage == NULL)
		{
			return false;
		}

		// Stick it in the array
		CIMAGE_MANAGER_ANIMATIONS.addElement(new CAnimationReference(theImage, referenceName, frameHeight, numberOfFrames));

		// Done :)
		return true;
	}
	catch(...)
	{
		return false;
	}
}

//	===========================================================================
bool CImageManager::readFromXML(const CSystemString &filename)
{
	// Create and read the document
	CXMLDocument document;
	if (!document.readFile(filename))
	{
		return false;
	}

	// Get the root node
	const CXMLNode *root = document.getRootNode();

	// Check its valid
	if (root && root->getNodeName() == CIMAGE_MANAGER_XML_TAG_ROOT)
	{
		const CXMLNode *images	   = root->getConstChildNode(CIMAGE_MANAGER_XML_TAG_IMAGES);
		const CXMLNode *animations = root->getConstChildNode(CIMAGE_MANAGER_XML_TAG_ANIMATIONS);

		// Now load the images
		if (images)
		{
			// Get number of images
			const long numberOfImages = images->getNumberOfChildNodes();

			// Loop through and get each one
			for (long i = 0; i < numberOfImages; i++)
			{
				// Get he image
				const CXMLNode *image = images->getConstChildNode(i);

				// Check its valid
				if (image == NULL)
				{
					continue;
				}

				// Get our attributes
				const CXMLAttribute *relativePath  = image->getConstAttribute(CIMAGE_MANAGER_XML_ATTRIBUTE_IMAGE_RELATIVE_PATH);
				const CXMLAttribute *filename      = image->getConstAttribute(CIMAGE_MANAGER_XML_ATTRIBUTE_IMAGE_FILENAME);
				const CXMLAttribute *referenceName = image->getConstAttribute(CIMAGE_MANAGER_XML_ATTRIBUTE_IMAGE_NAME);

				// Now add the image
				if (!addImage(CSystemString(relativePath->getValue()), filename->getValue(), referenceName->getValue()))
				{
					CTrace::trace(filename->getValue(), "Failed to load image with filename ");
				}
			}
		}

		// Now load the animations
		if (animations)
		{
			const long numberOfAnimations = animations->getNumberOfChildNodes();

			// Loop through and get numberOfAnimations one
			for (long i = 0; i < numberOfAnimations; i++)
			{
				// Get he image
				const CXMLNode *animation = animations->getConstChildNode(i);

				// Check its valid
				if (animation == NULL)
				{
					continue;
				}

				// Get our attributes
				const CXMLAttribute *relativePath  = animation->getConstAttribute(CIMAGE_MANAGER_XML_ATTRIBUTE_IMAGE_RELATIVE_PATH);
				const CXMLAttribute *filename      = animation->getConstAttribute(CIMAGE_MANAGER_XML_ATTRIBUTE_IMAGE_FILENAME);
				const CXMLAttribute *referenceName = animation->getConstAttribute(CIMAGE_MANAGER_XML_ATTRIBUTE_IMAGE_NAME);
				const CXMLAttribute *height        = animation->getConstAttribute(CIMAGE_MANAGER_XML_ATTRIBUTE_ANIMATION_FRAME_HEIGHT);
				const CXMLAttribute *numFrames     = animation->getConstAttribute(CIMAGE_MANAGER_XML_ATTRIBUTE_ANIMATION_NUM_FRAMES);

				// Now add the image
				if (!addAnimation(CSystemString(relativePath->getValue()), filename->getValue(), referenceName->getValue(), height->getValueAsLong(), numFrames->getValueAsLong()))
				{
					CTrace::trace(filename->getValue(), "Failed to load animation with filename ");
				}
			}
		}

		// We have loaded what was asked
		return true;
	}

	return false;
}

//	===========================================================================
bool CImageManager::loadFromDefinitionFile(const CSystemString &filename)
{
	try
	{
		// The stream that we read from
		CTextStream stream;

		// Open the stream
		if (!stream.openStream(filename, CTextStream::e_input))
		{
			return readFromXML(filename);
		}

		// The buffer to read in to
		CString buffer;
		CSystemString relativePath;
		CSystemString filename;
		CSystemString referenceName;
		CString frameHeight;
		CString numberOfFrames;
		bool animationState = false;		// Are we loading animations
		bool imageState     = false;		// Are we loading images
		bool failure		= false;		// Did anything fail to load

		// Find the head of the document
		stream >> buffer;
		if (buffer != "<start_skin>")
		{
			return readFromXML(filename);
		}

		// Now we want to keep reading...
		while(!stream.hasReachedEOF())
		{
			// Read in to the buffer
			stream >> buffer;

			// Remove leading white space
			buffer.removeLeadingWhiteSpace();
			buffer.removeTrailingWhiteSpace();

			//Check if its a comment
			if (buffer.characterAt(0) == '#')
			{
				continue;
			}

			// Found the end!
			if (buffer == "<end_skin>")
			{
				break;
			}

			// Check if we are in the images state
			if (buffer == "<images>")
			{
				imageState     = true;
				animationState = false;
				continue;
			}
			else if (buffer == "<animations>")
			{
				imageState     = false;
				animationState = true;
				continue;
			}

			// Now we handle the part based upon what we are
			if (imageState)
			{
				// RelativePath, Filename, ReferenceName
				CStringTokeniser tokeniser(buffer, ',');

				// Check we've got all the tokens
				if (tokeniser.getNumberOfTokens() != 3)
				{
					CTrace::trace(buffer, "Failed to load image with buffer ");
					failure = true;
					continue;
				}

				// Get the relative path
				if (!tokeniser.getNextToken(relativePath))
				{
					CTrace::trace(buffer, "Failed to load image with buffer ");
					failure = true;
					continue;
				}

				// Get the filename
				if (!tokeniser.getNextToken(filename))
				{
					CTrace::trace(buffer, "Failed to load image with buffer ");
					failure = true;
					continue;
				}

				// Get the reference name
				if (!tokeniser.getNextToken(referenceName))
				{
					CTrace::trace(buffer, "Failed to load image with buffer ");
					failure = true;
					continue;
				}

				// remove all the spaces
				relativePath.removeLeadingWhiteSpace();
				filename.removeLeadingWhiteSpace();
				referenceName.removeLeadingWhiteSpace();

				//CTrace::trace(filename, "Loading filename : ");

				// Now add
				if (!addImage(relativePath, filename, referenceName))
				{
					CTrace::trace(filename, "Failed to load image with filename ");
					failure = true;
					continue;
				}
			}
			else if (animationState)
			{
				// RelativePath, Filename, ReferenceName, FrameHeight, NumberOfFrames
				CStringTokeniser tokeniser(buffer, ',');

				// Check we've got all the tokens
				if (tokeniser.getNumberOfTokens() != 5)
				{
					CTrace::trace(buffer, "Failed to load animation with buffer ");
					failure = true;
					continue;
				}

				// Get the relative path
				if (!tokeniser.getNextToken(relativePath))
				{
					CTrace::trace(buffer, "Failed to load animation with buffer ");
					failure = true;
					continue;
				}

				// Get the filename
				if (!tokeniser.getNextToken(filename))
				{
					CTrace::trace(buffer, "Failed to load animation with buffer ");
					failure = true;
					continue;
				}

				// Get the reference name
				if (!tokeniser.getNextToken(referenceName))
				{
					CTrace::trace(buffer, "Failed to load animation with buffer ");
					failure = true;
					continue;
				}

				// Get the frame height
				if (!tokeniser.getNextToken(frameHeight))
				{
					CTrace::trace(buffer, "Failed to load animation with buffer ");
					failure = true;
					continue;
				}

				// Get the number of frames
				if (!tokeniser.getNextToken(numberOfFrames))
				{
					CTrace::trace(buffer, "Failed to load animation with buffer ");
					failure = true;
					continue;
				}

				// remove all the spaces
				relativePath.removeLeadingWhiteSpace();
				filename.removeLeadingWhiteSpace();
				referenceName.removeLeadingWhiteSpace();
				frameHeight.removeLeadingWhiteSpace();
				numberOfFrames.removeLeadingWhiteSpace();

				// Now add
				if (!addAnimation(relativePath, filename, referenceName, CString::toLong(frameHeight.getString()), CString::toLong(numberOfFrames.getString())))
				{
					CTrace::trace(buffer, "Failed to load animation with buffer ");
					failure = true;
					continue;
				}
			}
			else
			{
				CTrace::trace(buffer, "Unknown line with buffer ");
				continue;
			}
		}
		// Sucess!
		return !failure;
	}
	catch(...)
	{
		CTrace::trace("Caught unknown exception in the ImageManager::loadFromDefinitionFile function");
		return false;
	}
	return false;
}

//	===========================================================================
IImage *CImageManager::getImage(const CString &referenceName)
{
	for (long i = 0; i < CIMAGE_MANAGER_IMAGES.getInsertIndex(); i++)
	{
		// Get the reference
		CImageReference *reference = CIMAGE_MANAGER_IMAGES.elementAtIndex(i);

		// Check its valid
		if (reference && referenceName == reference->getReferenceName())
		{
			return reference->getImage();
		}
	}
	return NULL;
}

//	===========================================================================
CString CImageManager::getReferenceNameForImage(const IImage *image)
{
	if (image == NULL)
	{
		return CString::CSTRING_EMPTY_STRING;
	}

	for (long i = 0; i < CIMAGE_MANAGER_IMAGES.getInsertIndex(); i++)
	{
		// Get the reference
		CImageReference *reference = CIMAGE_MANAGER_IMAGES.elementAtIndex(i);

		if (reference && reference->getImage() == image)
		{
			return reference->getReferenceName();
		}
	}

	for (long i = 0; i < CIMAGE_MANAGER_IMAGES.getInsertIndex(); i++)
	{
		// Get the reference
		CAnimationReference *reference = CIMAGE_MANAGER_ANIMATIONS.elementAtIndex(i);

		// Check its valid
		if (reference && reference->getImage() == image)
		{
			return reference->getReferenceName();
		}
	}

	return CString::CSTRING_EMPTY_STRING;
}

//	===========================================================================
IImage *CImageManager::getAnimation(const CString &referenceName, long &frameHeight, long &numberOfFrames)
{
	for (long i = 0; i < CIMAGE_MANAGER_IMAGES.getInsertIndex(); i++)
	{
		// Get the reference
		CAnimationReference *reference = CIMAGE_MANAGER_ANIMATIONS.elementAtIndex(i);

		// Check its valid
		if (reference && referenceName == reference->getReferenceName())
		{
			reference->getAnimationSizes(frameHeight, numberOfFrames);
			return reference->getImage();
		}
	}
	return NULL;
}

//	===========================================================================
CImageManager::CAnimationReference *CImageManager::getAnimation(const CString &referenceName)
{
	for (long i = 0; i < CIMAGE_MANAGER_IMAGES.getInsertIndex(); i++)
	{
		// Get the reference
		CAnimationReference *reference = CIMAGE_MANAGER_ANIMATIONS.elementAtIndex(i);

		// Check its valid
		if (reference && referenceName == reference->getReferenceName())
		{
			return reference;
		}
	}
	return NULL;
}


//	===========================================================================
void CImageManager::clearImageLibrary()
{
	CIMAGE_MANAGER_IMAGES.clearArray();
	CIMAGE_MANAGER_ANIMATIONS.clearArray();
}

//	===========================================================================
IImage *CImageManager::getImageOnPath(const CSystemString &path)
{
	if (path.hasExtension("jpg") || path.hasExtension("jpeg"))
	{
		return new CJPEGImage(path);
	}
	else if (path.hasExtension("tga"))
	{
		return new CTGAImage(path);
	}
	else if (path.hasExtension("png"))
	{
		return new CPNGImage(path);
	}
	return NULL;
}	

//	===========================================================================
CImageManager::CImageReference::CImageReference() : m_image(NULL)
{
	NULL_POINTER(m_image);
	EXPONENT_CLASS_CONSTRUCTION(CImageManager::CImageReference);
}

//	===========================================================================
CImageManager::CImageReference::CImageReference(IImage *image, const CString &referenceName) : m_image(NULL)
{
	EXPONENT_CLASS_CONSTRUCTION(CImageManager::CImageReference);
	NULL_POINTER(m_image);
	this->setImage(image);
	this->setReferenceName(referenceName);
}

//	===========================================================================
CImageManager::CImageReference::~CImageReference()
{
	EXPONENT_CLASS_DESTRUCTION(CImageManager::CImageReference);
	FORGET_COUNTED_OBJECT(m_image);
}

//	===========================================================================
CImageManager::CAnimationReference::CAnimationReference()
{
	m_frameHeight    = 0;
	m_numberOfFrames = 0;
}

//	===========================================================================
CImageManager::CAnimationReference::CAnimationReference(IImage *image, const CString &referenceName, const long frameHeight, const long numberOfFrames)
             : m_frameHeight(0)
			 , m_numberOfFrames(0)
{
	EXPONENT_CLASS_CONSTRUCTION(CImageManager::CAnimationReference);
	m_frameHeight    = 0;
	m_numberOfFrames = 0;
	this->setImage(image);
	this->setReferenceName(referenceName);
	this->setAnimationSizes(frameHeight, numberOfFrames);
}

//	===========================================================================
CImageManager::CAnimationReference::~CAnimationReference()
{
	EXPONENT_CLASS_DESTRUCTION(CImageManager::CAnimationReference);
}